<?php
/**
 * @file  multiotp.api.tools.php
 * @brief Tools to call multiOTP Enterprise Web API commands
 *
 * multiOTP API tools
 *
 * PHP 5.4.0 or higher is supported.
 *
 * @author    Andre Liechti, SysCo systemes de communication sa, <support@multiotp.com>
 * @version   5.9.6.7
 * @date      2023-09-22
 * @since     2013-02-10
 * @copyright (c) 2010-2023 SysCo systemes de communication sa
 * @copyright GNU Lesser General Public License
 *********************************************************************/

 /**
 * @brief   Call a REST service, REST authentication is done as for Amazon services
 *          (http://docs.aws.amazon.com/AWSECommerceService/latest/DG/RequestAuthenticationArticle.html)
 *          (http://randomdrake.com/2009/07/27/amazon-aws-api-rest-authentication-for-php-5/)
 *
 * @param   array   $call_array  Array of parameters
 * @retval  string  Content received from the RESET service
 */
if (!function_exists('CallApi')) {
  function CallApi(
    $call_array = array("script_uri" => "",
                        "secret"     => "",
                        "post_data"  => "",
                        "timeout"    => 30)
  ) {
    $script_uri = isset($call_array["script_uri"]) ? $call_array["script_uri"] : "";
    $secret     = isset($call_array["secret"])     ? $call_array["secret"]     : "";
    $post_data  = isset($call_array["post_data"])  ? $call_array["post_data"]  : "";
    $timeout    = isset($call_array["timeout"])    ? $call_array["timeout"]    : 30;

    // Get a nice array of elements to work with
    $uri_elements = parse_url($script_uri);

    // Grab our request elements
    $scheme  = isset($uri_elements['scheme']) ? $uri_elements['scheme'] : 'http';
    $port    = isset($uri_elements['port'])   ? $uri_elements['port']   : '';
    $request = isset($uri_elements['query'])  ? $uri_elements['query']  : '';
    $host    = isset($uri_elements['host'])   ? $uri_elements['host']   : '';
    $path    = isset($uri_elements['path'])   ? $uri_elements['path']   : '';

    // Throw them into an array
    parse_str($request, $parameters);
    // $parameters = $_GET;

    if (isset($parameters['Signature'])) {
      unset($parameters['Signature']);
    }
    if (isset($parameters['Timestamp'])) {
      unset($parameters['Timestamp']);
    }
    $parameters['Timestamp'] = gmdate("Y-m-d\TH:i:s\Z");

    ksort($parameters);

    $request_array = array();
    // Create our new request
    foreach ($parameters as $parameter => $value) {
      // We need to be sure we properly encode the value of our parameter
      $parameter = str_replace("%7E", "~", rawurlencode($parameter));
      $value = str_replace("%7E", "~", rawurlencode($value));
      $request_array[] = $parameter . '=' . $value;
    }   

    // Put our & symbol at the beginning of each of our request variables and put it in a string
    $new_request = implode('&', $request_array);

    // Create our signature string
    $signature_string = "GET\n{$host}\n{$path}\n{$new_request}";

    // Create our signature using hash_hmac
    $signature = urlencode(base64_encode(hash_hmac('sha256', $signature_string, $secret, TRUE)));

    // Return our new request
    $url_request = "{$scheme}://{$host}".(('' != $port) ? ":{$port}" : "")."{$path}?{$new_request}&Signature={$signature}";

    // Use a pure PHP standalone HTTP request function
    $api_result = PostHttpDataXmlRequest(
      $post_data,   // $xml_data
      $url_request, // $xml_urls
      $timeout      // $xml_timeout
    );

    return $api_result;
  }
}


/**
 * @brief   Pure PHP standalone HTTP(S) request function
 *
 * @param   string  $xml_data           Complete data to be posted
 *          string  $xml_urls           Urls where to post, post to the next one in case of an error (separated by ;)
 *          string  $xml_timeout        Timeout before changing to the next server
 *          string  $xml_urls_splitter  String splitter between two Urls (default is ;)
 * @retval  string  Content received from the server (must contain <multiOTP to be valid)
 */
if (!function_exists('PostHttpDataXmlRequest')) {
  function PostHttpDataXmlRequest(
    $xml_data,
    $xml_urls,
    $xml_timeout = 3,
    $xml_urls_splitter = ";"
  ) {
    $result = FALSE;
    $content_to_post = 'data='.$xml_data;
    
    // Generic cleaner of multiple URLs
    $cleaned_xml_urls = trim(str_replace(" ",$xml_urls_splitter,str_replace(",",$xml_urls_splitter,str_replace(";",$xml_urls_splitter,$xml_urls))));
    $xml_url = explode($xml_urls_splitter,$cleaned_xml_urls);
    
    foreach ($xml_url as $xml_url_one) {
        
      $port = 80;

      $pos = mb_strpos($xml_url_one, '://');
      if (FALSE === $pos) {
        $protocol = '';
      } else {
        switch (mb_strtolower(mb_substr($xml_url_one,0,$pos),'UTF-8')) {
          case 'https':
          case 'ssl':
            $protocol = 'ssl://';
            $port = 443;
            break;
          case 'tls':
            $protocol = 'tls://';
            $port = 443;
            break;
          default:
            $protocol = '';
            break;
        }
        
        $xml_url_one = mb_substr($xml_url_one,$pos+3);
      }
      
      $pos = mb_strpos($xml_url_one, '/');
      if (FALSE === $pos) {
        $host = $xml_url_one;
        $url = '/';
      } else {
        $host = mb_substr($xml_url_one,0,$pos);
        $url = mb_substr($xml_url_one,$pos); // And not +1 as we want the / at the beginning
      }
      
      $pos = mb_strpos($host, ':');
      if (FALSE !== $pos) {
        $port = mb_substr($host,$pos+1);
        $host = mb_substr($host,0,$pos);
      }
      
      $errno = 0;
      $errdesc = 0;
      if (function_exists("stream_socket_client")) {
        $sslContext = stream_context_create(
          array('ssl' => array('verify_peer'         => false,
                               'verify_peer_name'    => false,
                               'disable_compression' => true,
                               'ciphers'             => 'ALL!EXPORT!EXPORT40!EXPORT56!aNULL!LOW!RC4'
                              )
               )
        );
        $fp = @stream_socket_client($protocol.$host.":".$port, $errno, $errdesc, $xml_timeout, STREAM_CLIENT_CONNECT, $sslContext);
      } else {
        $fp = @fsockopen($protocol.$host, $port, $errno, $errdesc, $xml_timeout);
      }
      
      if (FALSE !== $fp) {
        $info['timed_out'] = FALSE;
        fputs($fp, "POST ".$url." HTTP/1.0\r\n");
        fputs($fp, "Content-Type: application/x-www-form-urlencoded\r\n");
        fputs($fp, "Content-Length: ".mb_strlen($content_to_post)."\r\n");
        fputs($fp, "User-Agent: multiOTP\r\n");
        fputs($fp, "Host: ".$host."\r\n");
        fputs($fp, "\r\n");
        fputs($fp, $content_to_post);
        fputs($fp, "\r\n");

        // At least 20 seconds before the stream timeout (the server can be reached if we are here)
        $stream_timeout = $xml_timeout;
        if ($stream_timeout < 20) {
          $stream_timeout = 20;
        }

        stream_set_blocking($fp, TRUE);
        stream_set_timeout($fp, $stream_timeout);
        $info = stream_get_meta_data($fp); 

        $reply = '';
        $last_length = 0;
        while ((!feof($fp)) && ((!$info['timed_out']) || ($last_length != mb_strlen($reply)))) {
          $last_length = mb_strlen($reply);
          $reply.= fgets($fp, 1024);
          $info = stream_get_meta_data($fp);
        }
        fclose($fp);

        if (!$info['timed_out']) {
          $pos = mb_strpos(mb_strtolower($reply,'UTF-8'), "\r\n\r\n");
          $header = mb_substr($reply, 0, $pos);
          $answer = mb_substr($reply, $pos + 4);
          $header_array = explode(" ", $header."   ");
          $status = intval($header_array[1]);
          
          $result = $answer;

          if (FALSE !== mb_strpos($result, '<multiOTP')) {
            break; // Break of the foreach loop
          }
        }
        // If we are here, something was bad with the actual server
      }
    } // foreach

    return $result;
  }
}


/***********************************************************************
 * Custom function nice_json
 *
 * http://stackoverflow.com/a/9776726
 *
 * @author Kendall Hopkins
 ***********************************************************************/
if (!function_exists('nice_json')) {
  function nice_json($json, $separator = "\t")
  {
    $result = '';
    $level = 0;
    $in_quotes = false;
    $in_escape = false;
    $ends_line_level = NULL;
    $json_length = strlen( $json );

    for( $i = 0; $i < $json_length; $i++ ) {
      $char = $json[$i];
      $new_line_level = NULL;
      $post = "";
      if( $ends_line_level !== NULL ) {
        $new_line_level = $ends_line_level;
        $ends_line_level = NULL;
      }
      if ( $in_escape ) {
        $in_escape = false;
      } elseif( $char === '"' ) {
        $in_quotes = !$in_quotes;
      } elseif( ! $in_quotes ) {
        switch( $char ) {
          case '}': case ']':
            $level--;
            $ends_line_level = NULL;
            $new_line_level = $level;
            break;

          case '{':
          case '[':
            $level++;
          case ',':
            $ends_line_level = $level;
            break;

          case ':':
            $post = " ";
            break;

          case " ":
          case "\t":
          case "\n":
          case "\r":
            $char = "";
            $ends_line_level = $new_line_level;
            $new_line_level = NULL;
            break;
        }
      } elseif ( $char === '\\' ) {
        $in_escape = true;
      }
      if( $new_line_level !== NULL ) {
        $result .= "\n".str_repeat($separator, $new_line_level);
      }
      $result .= $char.$post;
    }

    return $result;
  }
}